<?php
require 'db_config.php';

header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");

$data = json_decode(file_get_contents("php://input"));

// Validate input
if (!isset($data->telegram_id) || !is_numeric($data->telegram_id) || !isset($data->task_id) || !is_numeric($data->task_id)) {
    echo json_encode(['success' => false, 'message' => 'Invalid or missing parameters.']);
    exit();
}

$telegram_id = (int)$data->telegram_id;
$task_id = (int)$data->task_id;

// Check if user has already completed this task
$stmt_check = $conn->prepare("SELECT id FROM user_completed_tasks WHERE user_telegram_id = ? AND task_id = ?");
$stmt_check->bind_param("ii", $telegram_id, $task_id);
$stmt_check->execute();
if ($stmt_check->get_result()->num_rows > 0) {
    echo json_encode(['success' => false, 'message' => 'You have already completed this task.']);
    $stmt_check->close();
    $conn->close();
    exit();
}
$stmt_check->close();

// Start transaction
$conn->begin_transaction();

try {
    // Get task reward
    $stmt_task = $conn->prepare("SELECT reward FROM tasks WHERE id = ? AND is_active = 1");
    $stmt_task->bind_param("i", $task_id);
    $stmt_task->execute();
    $result_task = $stmt_task->get_result();
    
    if ($result_task->num_rows === 0) {
        throw new Exception("Task not found or is inactive.");
    }
    
    $task = $result_task->fetch_assoc();
    $reward = (float)$task['reward'];

    // Update user's balance
    $stmt_update = $conn->prepare(
        "UPDATE users SET 
            balance = balance + ?, 
            total_earnings = total_earnings + ? 
        WHERE telegram_id = ?"
    );
    $stmt_update->bind_param("ddi", $reward, $reward, $telegram_id);
    if (!$stmt_update->execute()) {
        throw new Exception("Failed to update user balance.");
    }

    // Mark task as completed for the user
    $stmt_insert = $conn->prepare("INSERT INTO user_completed_tasks (user_telegram_id, task_id) VALUES (?, ?)");
    $stmt_insert->bind_param("ii", $telegram_id, $task_id);
    if (!$stmt_insert->execute()) {
        throw new Exception("Failed to record task completion.");
    }

    // Commit transaction
    $conn->commit();

    echo json_encode(['success' => true, 'message' => 'Task completed! Reward of ' . $reward . ' added.']);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

$conn->close();
?>