<?php
require 'db_config.php';

header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Read the incoming JSON data from the request body
$data = json_decode(file_get_contents("php://input"));

// Validate input
if (!isset($data->telegram_id) || !is_numeric($data->telegram_id)) {
    echo json_encode(['success' => false, 'message' => 'Invalid or missing Telegram ID.']);
    exit();
}

$telegram_id = (int)$data->telegram_id;

// Get settings from the database
$ads_reward = (float)get_setting($conn, 'ads_reward');
$daily_ads_limit = (int)get_setting($conn, 'daily_ads_limit');

// Start transaction for data consistency
$conn->begin_transaction();

try {
    // Lock the user row to prevent race conditions
    $stmt = $conn->prepare("SELECT ads_watched_today, last_ad_watched_date FROM users WHERE telegram_id = ? FOR UPDATE");
    $stmt->bind_param("i", $telegram_id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        throw new Exception("User not found.");
    }

    $user = $result->fetch_assoc();
    $ads_watched_today = $user['ads_watched_today'];
    $last_ad_watched_date = $user['last_ad_watched_date'];
    $today = date('Y-m-d');

    // Reset daily count if the date has changed
    if ($last_ad_watched_date < $today) {
        $ads_watched_today = 0;
    }

    // Check if user has reached the daily limit
    if ($ads_watched_today >= $daily_ads_limit) {
        throw new Exception("Daily ad limit reached. Please come back tomorrow.");
    }

    // Update user's balance and ad count
    $update_stmt = $conn->prepare(
        "UPDATE users SET 
            balance = balance + ?, 
            total_earnings = total_earnings + ?, 
            ads_watched_today = ?,
            last_ad_watched_date = ?
        WHERE telegram_id = ?"
    );
    
    $new_ads_count = $ads_watched_today + 1;
    $update_stmt->bind_param("ddisi", $ads_reward, $ads_reward, $new_ads_count, $today, $telegram_id);
    
    if (!$update_stmt->execute()) {
        throw new Exception("Failed to update earnings.");
    }

    // Commit the transaction
    $conn->commit();

    echo json_encode([
        'success' => true, 
        'message' => 'Reward of ' . $ads_reward . ' has been added successfully!'
    ]);

} catch (Exception $e) {
    // Rollback the transaction on error
    $conn->rollback();
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

$conn->close();
?>