<?php
require 'db_config.php';

header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");

$data = json_decode(file_get_contents("php://input"));

// Basic validation
if (
    !isset($data->telegram_id) || !is_numeric($data->telegram_id) ||
    !isset($data->amount) || !is_numeric($data->amount) || $data->amount <= 0 ||
    !isset($data->method) || empty(trim($data->method)) ||
    !isset($data->account_number) || empty(trim($data->account_number))
) {
    echo json_encode(['success' => false, 'message' => 'Please fill all fields correctly.']);
    exit();
}

$telegram_id = (int)$data->telegram_id;
$amount = (float)$data->amount;
$method = trim($data->method);
$account_number = trim($data->account_number);

// Start transaction
$conn->begin_transaction();

try {
    // Check user's balance (lock the row for update)
    $stmt_balance = $conn->prepare("SELECT balance FROM users WHERE telegram_id = ? FOR UPDATE");
    $stmt_balance->bind_param("i", $telegram_id);
    $stmt_balance->execute();
    $result_balance = $stmt_balance->get_result();

    if ($result_balance->num_rows === 0) {
        throw new Exception("User not found.");
    }

    $user = $result_balance->fetch_assoc();
    $current_balance = (float)$user['balance'];

    // Check if balance is sufficient
    if ($current_balance < $amount) {
        throw new Exception("Insufficient balance for this withdrawal request.");
    }

    // Deduct the amount from user's balance
    $new_balance = $current_balance - $amount;
    $stmt_deduct = $conn->prepare("UPDATE users SET balance = ? WHERE telegram_id = ?");
    $stmt_deduct->bind_param("di", $new_balance, $telegram_id);
    if (!$stmt_deduct->execute()) {
        throw new Exception("Failed to update balance.");
    }

    // Insert the withdrawal request into the withdrawals table
    $stmt_insert = $conn->prepare(
        "INSERT INTO withdrawals (user_telegram_id, method, account_number, amount) 
        VALUES (?, ?, ?, ?)"
    );
    $stmt_insert->bind_param("issd", $telegram_id, $method, $account_number, $amount);
    if (!$stmt_insert->execute()) {
        throw new Exception("Failed to create withdrawal request.");
    }
    
    // If all successful, commit the transaction
    $conn->commit();

    echo json_encode(['success' => true, 'message' => 'Withdrawal request submitted successfully. It will be processed soon.']);

} catch (Exception $e) {
    // If any error, rollback the entire transaction
    $conn->rollback();
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

$conn->close();
?>```